space      [ \t]
white      [ \n\t]
return     [\n]
ident      [A-Za-z_][_A-Za-z0-9]*
tagident   [A-Za-z_][_A-Za-z0-9,]*
namechar   [^<>(){}#,;%/]
paramchar  [^<>(){}#;%/]
typechar   [^;#%{}]
typedef    {space}*"\n"?*{space}*"typedef"{white}+[^{};]*
typesub    "\n"{space}*"/**"
typesub2   "\n"{typechar}+";"{space}*"/**"
define     {space}*"\n#"{space}*"define"{space}+{ident}
macro      {space}*"\n#"{space}*"define"{space}+{ident}"("{paramchar}*")"
function   {space}*"\n"{space}*{namechar}+"("{paramchar}*")"{space}*
global     {space}*"\n"{space}*"extern"{space}+{namechar}+";\n"
plaindocs  "/**"{white}+
taggeddocs "/**"{tagident}
tagafter   {white}+"!"?
ccomment   "/*"

%array
%option yylineno
%x INDOC AFTERDOC
%x INTYPEDEF INTYPENAME TYPESUBDATA
%x INDEFINE INDEFDATA
%x INFUNC

%{
#define MXDOC_VERSION_STRING "2.6"

/*
    MXDOC
    HTML documentation generator for C
    Copyright (C) 2006 Douglas Eleveld

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* NOTES:

 - The first non-connected documentation in a module is the assumed the main
   documentation for the module.  

 - Whitespace is maintained within docs except for leading and trailing
   whitespace.

 - Documentation from two places with exactly the same text (i.e. from a
   function delaration and definition) are combined.  The entire text must
   match exactly *even parameter names*.
   
 - If docs for function definition are added to those from function
   declaration (because they share the same name) then the docs from the
   definition are added as the first subdocs.
   
*/

#include <stdlib.h>    
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <values.h>
#include <time.h>
#include "deds/maxmin.h"
#include "deds/vector/vector.c"
#include "deds/vector/free.c"
#include "deds/vector/makeroom.c"
#include "deds/vector/reserve.c"
#include "deds/vector/remove.c"
#include "deds/vector/resize.c"
#include "deds/tree.h"

typedef MX_VECTOR(char) STRING;

typedef struct DNODE {
    STRING name;
    STRING docs;
    STRING tags;
    STRING include;
    MX_TREE(struct DNODE);
} DNODE;

static DNODE mainpage;
static DNODE current;

static STRING filename;
static STRING modname;
static STRING libname;
static STRING alltags;
static int brace = 0;
static int bracelevel = 0;
static int subbracelevel = 0;
static time_t now;

#define append_str(a,t) mx_vector_append((a), (t), strlen(t))
#define append_vec(d,s) mx_vector_append((d), (s)->data, mx_vector_size(s))

static void readdocs(STRING* string);
static void readtags(STRING* string);

static void sprint(const STRING* string, FILE* stream, unsigned asheader)
{
    unsigned i;

    if (asheader)
        fputs("<h2>", stream);
    
    for (i=0; i<mx_vector_size(string); i++) {
        if ((asheader) && (string->data[i] == '\n')) {
            fputs("</h2><p>", stream);
            asheader = 0;
        }
        
        assert(string->data[i]);
        fputc(string->data[i], stream);
        fflush(stream);
    }

    if (asheader)
        fputs("</h2><p>", stream);
}

static DNODE* dnode(DNODE* parent, STRING* name)
{
    assert(parent);
    DNODE* ptr = mx_tree_first(parent);

    if (name) {
        const unsigned len = mx_vector_size(name);
    
        while (ptr) {
            if ((len == mx_vector_size(&ptr->name)) &&
                (strncmp(name->data, ptr->name.data, len) == 0))
                return ptr;

            ptr = mx_tree_next(ptr);
        }
    }
    
    ptr = malloc(sizeof(DNODE));
    memset(ptr, 0, sizeof(DNODE));
    if (name)
        append_vec(&ptr->name, name);
    MX_TREE_INSERT(ptr, parent);

    return ptr;
}

static DNODE* makelib(STRING* libname)
{
    assert(libname);
    return dnode(&mainpage, libname);
}

static DNODE* makemod(STRING* libname, STRING* modname)
{
    assert(libname);
    assert(modname);

    if ((mx_vector_size(libname) == mx_vector_size(modname)) &&
        (strncmp(libname->data, modname->data, mx_vector_size(libname) == 0)))
        return makelib(libname);
    
    return dnode(makelib(libname), modname);
}

static DNODE* makenode(STRING* libname, STRING* modname, STRING* name)
{
    if (mx_vector_size(name) == 0)
        name = 0;

    return dnode(makemod(libname, modname), name);
}

static unsigned matchtag(STRING* tags, const char* text, const int n)
{
    STRING temp;
    int ret = 0;

    if ((!text) || (text[0] == 0) || (text[0] == ',') || (n == 0))
        return 1;
    
    mx_vector(&temp);
    mx_vector_append1(&temp, ',');
    mx_vector_append(&temp, text, n);
    mx_vector_append1(&temp, ',');
    
    mx_vector_append1(&temp, 0);
    mx_vector_append1(tags, 0);

    if (strstr(tags->data, temp.data))
        ret = 1;

    mx_vector_remove_last(tags);
    mx_vector_free(&temp);

    return ret;
}

static void whiteclean(STRING* string)
{
    while ((mx_vector_size(string)) && (isspace(string->data[0])))
        mx_vector_remove(string, 0, 1);

    while ((mx_vector_size(string)) && (isspace(string->data[mx_vector_size(string) - 1])))
        mx_vector_remove(string, mx_vector_size(string) - 1, 1);
}

static void addtags(STRING* tags, const char* text)
{
    int n;
    const char * p;

    while ((*text == ',') || (*text == ' ') || (*text == '\t') || (*text == '\n'))
        ++text;
    
    do {
        p = strchr(text, ',');
        n = (p) ? (p - text) : (int)strlen(text);

        if (matchtag(tags, text, n) == 0) {
             if (mx_vector_size(tags) == 0)
                 mx_vector_append1(tags, ',');
             mx_vector_append(tags, text, n);
             mx_vector_append1(tags, ',');
        }

        text = p + 1;
    } while ((p) && (*p));
}

static void namefixup(STRING* string)
{
    unsigned i, changed;

    for (i=0; i<mx_vector_size(string); i++)
        if (isspace(string->data[i]))
            string->data[i] = ' ';

    do {
        changed = 0;
            
        for (i=1; i<mx_vector_size(string); i++) {
            if ((string->data[i - 1] == ' ') && (string->data[i] == ',')) {
                string->data[i - 1] = ',';
                string->data[i] = ' ';
                changed = 1;
            }
        }

        for (i=1; i<mx_vector_size(string); i++) {
            if ((string->data[i - 1] == ' ') && (string->data[i] == '*')) {
                string->data[i - 1] = '*';
                string->data[i] = ' ';
                changed = 1;
            }
        }
        
        for (i=1; i<mx_vector_size(string); i++) {
            if ((string->data[i - 1] == ' ') && (string->data[i] == ' ')) {
                mx_vector_remove(string, i, 1);
                changed = 1;
            }
        }
        
    } while (changed);
}

static void specialtags(STRING* tags, STRING* docs)
{
    /* These checks have to be done repetitively in case a doc has more than
       one of these tags at the same time */
    if (matchtag(tags, "mainpage", strlen("mainpage"))) {
        DNODE* ptr = &mainpage;
        if (mx_vector_size(&ptr->docs))
            append_str(&ptr->docs, "\n<p>\n");
        append_vec(&ptr->docs, docs);
    }
    
    if (matchtag(tags, "library", strlen("library"))) {
        DNODE* ptr = makelib(&libname);
        if (mx_vector_size(&ptr->docs))
            append_str(&ptr->docs, "\n<p>\n");
        append_vec(&ptr->docs, docs);
    }
    
    if (matchtag(tags, "module", strlen("module"))) {
        DNODE* ptr = makemod(&libname, &modname);
        if (mx_vector_size(&ptr->docs))
            append_str(&ptr->docs, "\n<p>\n");
        append_vec(&ptr->docs, docs);
    }
}

static void finalize(void)
{
    whiteclean(&current.name);
    whiteclean(&current.docs);
    whiteclean(&current.tags);
    namefixup(&current.name);

    if ((mx_vector_size(&current.docs))) {
        DNODE *dnode = makenode(&libname, &modname, &current.name);

        if (mx_vector_size(&dnode->docs))
            append_str(&dnode->docs, "<br>\n");
        append_vec(&dnode->docs, &current.docs);
        mx_vector_append1(&current.tags, 0);
        addtags(&dnode->tags, current.tags.data);
        addtags(&alltags, current.tags.data);

        specialtags(&current.tags, &current.docs);
    }
    
    mx_vector_resize(&current.name, 0);
    mx_vector_resize(&current.docs, 0);
    mx_vector_resize(&current.tags, 0);
}

static void finalizetagdocs(STRING* docs, const char* tagtext)
{
    whiteclean(docs);
    
    DNODE *dnode = makenode(&libname, &modname, 0);

    append_vec(&dnode->docs, docs);
    addtags(&dnode->tags, tagtext);
    addtags(&alltags, tagtext);

    specialtags(&dnode->tags, &dnode->docs);
}

%}

%%
<INITIAL,AFTERDOC>{typedef}"{" { /* typedef with braces -------------------*/
                   bracelevel = brace++;
                   BEGIN INTYPEDEF;
                   addtags(&current.tags, "type");
                   addtags(&alltags, "type");
                 }
<AFTERDOC>{typedef}";" { /* A typedef with no { } ends at the semicolon */
                   BEGIN 0;
                   mx_vector_append(&current.name, yytext, yyleng);
                   addtags(&current.tags, "type");
                   addtags(&alltags, "type");
                   finalize();
                 }
<INTYPEDEF>"}"   { /* After the last brace the typename starts */
                   --brace;
                   if (brace == bracelevel)
                       BEGIN INTYPENAME;
                 }
<INTYPENAME>{ident} { /* the typedef name (more than one?) */
                   mx_vector_append(&current.name, yytext, yyleng);
                 }
<INTYPENAME>";"  { /* Typedef name continues till semicolon */
                   BEGIN 0;
                   finalize();
                 }

<INTYPEDEF>{plaindocs}/!?{ident} { /* typedef subdocs --------------------------------*/
                   BEGIN TYPESUBDATA;
                   subbracelevel = brace;
                   readdocs(&current.docs);
                 }
<TYPESUBDATA>";" { if (brace == subbracelevel)
                       BEGIN INTYPEDEF;
                   else
                       mx_vector_append(&current.name, yytext, yyleng);
                 }
<TYPESUBDATA>{white}+ {
                   append_str(&current.name, " ");
                 }
<TYPESUBDATA>.   {
                   append_str(&current.name, yytext);
                 }

<INTYPEDEF>{typesub} { /* typedef subdocs (prev) --------------------------*/
                   STRING temp, tag;
                   int c;
                   
                   mx_vector(&tag);
                   readtags(&tag);

                   mx_vector(&temp);
                   readdocs(&temp);

                   if (mx_vector_size(&current.docs))
                       append_str(&current.docs, "<p>");
                       
                   append_str(&current.docs, "member: <b>");

                   do {
                       c = input();
                       mx_vector_append1(&current.docs, c);
                   } while (c != ';');
                   append_str(&current.docs, "</b> ");
                   
                   append_vec(&current.docs, &temp);

                   if (mx_vector_size(&tag)) {
                       mx_vector_append1(&tag, 0);
                       finalizetagdocs(&temp, tag.data);
                   }
                   
                   mx_vector_free(&temp);
                   mx_vector_free(&tag);
                 }

<INTYPEDEF>{typesub2} { /* typedef subdocs (next) --------------------------*/
                   STRING temp, tag;
                   const char* p = strchr(yytext, ';');
                   
                   mx_vector(&tag);
                   readtags(&tag);

                   mx_vector(&temp);
                   readdocs(&temp);

                   if (mx_vector_size(&current.docs))
                       append_str(&current.docs, "<p>");
                       
                   if (p) {
                       append_str(&current.docs, "member: <b>");
                       mx_vector_append(&current.docs, yytext, p - yytext);
                       append_str(&current.docs, "</b> ");
                   }
                   
                   append_vec(&current.docs, &temp);

                   if (mx_vector_size(&tag)) {
                       mx_vector_append1(&tag, 0);
                       finalizetagdocs(&temp, tag.data);
                   }
                   
                   mx_vector_free(&temp);
                   mx_vector_free(&tag);
                 }

<INITIAL,AFTERDOC>{define} { /* define -------------------------------------*/
                   BEGIN INDEFDATA;
                   addtags(&current.tags, "define");
                   addtags(&alltags, "define");
                   append_str(&current.name, yytext);
                 }
<INITIAL,AFTERDOC>{macro} { /* macro ---------------------------------------*/
                   BEGIN INDEFDATA;
                   addtags(&current.tags, "macro");
                   addtags(&alltags, "macro");
                   append_str(&current.name, yytext);
                 }
<INDEFDATA>.     { };
<INDEFDATA>\\\n  { /* Swallow escaped line endings */ };
<INDEFDATA>{plaindocs}/!?{ident} { /* define or macro subdocs ------------------------*/
                   readdocs(&current.docs);
                 }
<INDEFDATA>\n    { /* defines/macros end at real return */
                   BEGIN 0;
                   finalize();
                 }

<INITIAL,AFTERDOC>{function}"\n{" { /* function ----------------------------*/
                   bracelevel = brace++;
                   BEGIN INFUNC;
                   addtags(&current.tags, "function");
                   addtags(&alltags, "function");
                   mx_vector_append(&current.name, yytext, yyleng - 2);
                 }
<INITIAL,AFTERDOC>{function}";" { /* function declaratiion */
                   BEGIN 0;
                   addtags(&current.tags, "function");
                   addtags(&alltags, "function");
                   mx_vector_append(&current.name, yytext, yyleng - 1);
                   finalize();
                 }
<INFUNC>"}"      { /* Functions stop on a closing brace */
                   --brace;
                   if (brace == bracelevel) {
                       BEGIN 0;
                       finalize();
                   }
                 }

<INFUNC>{plaindocs}/!?{ident} { /* function subdocs ----------------------------------*/
                   readdocs(&current.docs);
                 }

<INITIAL>{plaindocs}/!?{ident} { /* documentation ------------------------------------*/
                   BEGIN AFTERDOC;
                   readdocs(&current.docs);
                 }
                 
<AFTERDOC>{plaindocs}/!?{ident} {
                   if (mx_vector_size(&current.tags) == 0) {
                       addtags(&current.tags, "module");
                       addtags(&alltags, "module");
                   }
                   finalize();
                   readdocs(&current.docs);
                 }
                 
<AFTERDOC>.|\n   { BEGIN 0;
                   if (mx_vector_size(&current.tags) == 0) {
                       addtags(&current.tags, "module");
                       addtags(&alltags, "module");
                   }
                   finalize();
                 }
<INITIAL>{taggeddocs}/{tagafter} { /* Tagged docs at base level mark what follows -----*/
                   addtags(&current.tags, yytext + 3);
                   addtags(&alltags, yytext + 3);
                   readdocs(&current.docs);
                   BEGIN AFTERDOC;
                 }
<AFTERDOC>{taggeddocs}/{tagafter} {
                   if (mx_vector_size(&current.tags) == 0) {
                       addtags(&current.tags, "module");
                       addtags(&alltags, "module");
                   }
                   finalize();
                   addtags(&current.tags, yytext + 3);
                   addtags(&alltags, yytext + 3);
                   readdocs(&current.docs);
                 }
                 
<AFTERDOC>{global} { /* global variable ----------------------------*/
                   BEGIN 0;
                   addtags(&current.tags, "global");
                   addtags(&alltags, "global");
                   mx_vector_append(&current.name, yytext, yyleng);
                   finalize();
                 }
                 
<*>{taggeddocs}/{tagafter} { /* Tagged docs within func/struct mark top level docs */
                   STRING temp;
                   mx_vector(&temp);
                   readdocs(&temp);

                   if (mx_vector_size(&current.docs))
                       append_str(&current.docs, "<p>");
                   append_vec(&current.docs, &temp);
                   
                   finalizetagdocs(&temp, yytext + 3);
                   mx_vector_free(&temp);
                 }
<*>{ccomment}    {
                   int c = ' ', last;
                   while (c != EOF) {
                       last = c;
                       c = input();
                       if ((c == '/') && (last == '*'))
                           break;
                   }
                 }
<*>"{"           { ++brace; }
<*>"}"           { --brace; }
<*>.|\n          { }

%%

static void readdocs(STRING* string)
{
    char cstr[3];
    int c = input();

    cstr[0] = c;
    cstr[1] = 0;
    cstr[2] = 0;

    while (isspace(c)) {
        c = input();
        cstr[0] = c;
    }

    if ((string) && (mx_vector_size(string) != 0))
        append_str(string, "\n<p>");

    while (1) {
        if (cstr[0] == '*') {
            cstr[1] = input();
            if (cstr[1] == '/')
                break;
        }

        if (string)
            append_str(string, cstr);
        
        cstr[0] = input();
        cstr[1] = 0;
    }

    if (string) {
        append_str(string, " \n(<a href=\"");
        append_vec(string, &filename);
        append_str(string, "\">");
        append_vec(string, &filename);
        append_str(string, "</a>)\n");
    }
}

static void readtags(STRING* string)
{
    int c = input();

    while ((c != ' ') && (c != '\t') && (c != '\n')) {
        mx_vector_append1(string, c);
        c = input();
    }
    unput(c);
}

int yywrap(void) { return 1; };

static void process_file(const char* fname)
{
    fprintf(stderr, "file: %s\n", fname);
    mx_vector_resize(&filename, 0);
    append_str(&filename, fname);

    BEGIN 0;
    brace = 0;
    bracelevel = 0;
    yyin = fopen(fname, "r");
    assert(yyin);
    yylex();
    finalize();
    fclose(yyin);
}

#include "findmod.c"

static FILE* htmlopen(const STRING* prefix, const STRING* title, unsigned toplink)
{
    STRING temp;

    mx_vector(&temp);
    if ((prefix) && (mx_vector_size(prefix))) {
        append_vec(&temp, prefix);
        append_str(&temp, "_");
    }
    append_vec(&temp, title);
    append_str(&temp, ".htm");
    mx_vector_append1(&temp, 0);

    fprintf(stderr, "Writing: %s\n", temp.data);

    FILE* stream = fopen(temp.data, "w");
    if (stream) {
        fputs("<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/1999/REC-html401-19991224/strict.dtd\">\n"
              "<html><head>\n"
              "<title>", stream);
        sprint(title, stream, 0);
        fputs("</title>\n</head><body>\n", stream);

        if (toplink) {
            fputs("<p>[<a href=\"mxdoc.htm\">mainpage</a>]\n", stream);
            fputs("[<a href=\"tags.htm\">tags</a>]<br>\n", stream);
        }
    }

    mx_vector_free(&temp);
    return stream;
}

static void htmlclose(FILE* stream)
{
    fputs("<p>Generated by <a href=\"http://www.deleveld.dds.nl/mxdoc/index.htm\">MXDOC</a> ", stream);
    fputs(MXDOC_VERSION_STRING, stream);
    fputs(" on ", stream);
    time(&now);
    fprintf(stream, "%s\n", asctime(localtime(&now)));
    
    fputs("\n</body></html>\n", stream);
    fclose(stream);
}

static void writedocs(const DNODE* node, FILE* stream, const int full)
{
    if (mx_vector_size(&node->docs)) {
        unsigned i;
        int intitle = 0;
        
        if ((node->docs.data[0] != '!') && (!full))
            return;
            
        if (node->docs.data[0] != '!') {
            fputs("<p>", stream);
            fputc(node->docs.data[0], stream);
            
        } else if (full) {
            fputs("<p><b>", stream);
            intitle = 1;
        }

        for (i=1; i<mx_vector_size(&node->docs); i++) {
            if ((!full) &&
                (node->docs.data[0] == '!') &&
                (node->docs.data[i] == '\n'))
                break;

            if ((intitle) &&
                (full) &&
                (node->docs.data[0] == '!') &&
                (node->docs.data[i] == '\n')) {
                fputs("</b><p>", stream);
                intitle = 0;
            }
                
            assert(node->docs.data[i]);
            fputc(node->docs.data[i], stream);
        }
        if ((full) && (intitle))
            fputs("</b>", stream);
    }
}

static void writeheader(const DNODE* node, FILE* stream, const char* header)
{
    if (mx_vector_size(&node->name)) {
        unsigned i;
        
        fputs("<h1><a name=\"", stream);
        if (node->name.data[0] != '#')
            fputc(node->name.data[0], stream);
        for (i=1; i<mx_vector_size(&node->name); i++)
            fputc(node->name.data[i], stream);
        fprintf(stream, "\">%s", header);
        sprint(&node->name, stream, 0);
        fprintf(stream, "</a></h1>");
    }

    writedocs(node, stream, 1);
}

static void writenode(const DNODE* node, FILE* stream)
{
    if (mx_vector_size(&node->name)) {
        unsigned i;
        
        fputs("<a name=\"", stream);
        if (node->name.data[0] != '#')
            fputc(node->name.data[0], stream);
        for (i=1; i<mx_vector_size(&node->name); i++)
            fputc(node->name.data[i], stream);
        fprintf(stream, "\"><code>");
        sprint(&node->name, stream, 0);
        fprintf(stream, "</code></a>");
    }

    if (mx_vector_size(&node->docs)) {
        if (mx_vector_size(&node->name))
            fputs("\n<p>\n", stream);
        writedocs(node, stream, 1);
    }
}

static void writeref(const DNODE* lib, const DNODE* mod, FILE* stream)
{
    fputs("<p>(library <a href=\"", stream);
    sprint(&lib->name, stream, 0);
    fputs(".htm\">", stream);
    sprint(&lib->name, stream, 0);
    fputs("</a> : module <a href=\"", stream);
    if (mx_vector_size(&lib->name)) {
        sprint(&lib->name, stream, 0);
        fputs("_", stream);
    }
    sprint(&mod->name, stream, 0);
    fputs(".htm\">", stream);
    sprint(&mod->name, stream, 0);
    fputs("</a>)\n", stream);
}

static unsigned substantialmodule(DNODE* mod)
{
    if (mx_vector_size(&mod->name) == 0)
        return 0;
         
    DNODE* node = mx_tree_first(mod);
    while (node) {
        if (mx_vector_size(&node->name))
            return 1;
        node = mx_tree_next(node);
    }
    return 0;
}

static void writemainpage(void)
{
    FILE* stream;
    STRING title;

    mx_vector(&title);
    append_str(&title, "mxdoc");
    
    stream = htmlopen(0, &title, 0);
    assert(stream);

    writenode(&mainpage, stream);

    int numlib = 0;
    DNODE* lib = mx_tree_first(&mainpage);
    while (lib) {
        if (mx_vector_size(&lib->name))
            ++numlib;
        lib = mx_tree_next(lib);
    }

    if (numlib) {
        fputs("<h2>libraries:</h2><ul>", stream);

        lib = mx_tree_first(&mainpage);
        while (lib) {
            if (mx_vector_size(&lib->name)) {
                fputs("<li>", stream);
                fputs("<a href=\"", stream);
                sprint(&lib->name, stream, 0);
                fputs(".htm", stream);
                fputs("\">", stream);
                sprint(&lib->name, stream, 0);
                fputs("</a> \n", stream);
                writedocs(lib, stream, 0);
                fputs("</li>\n", stream);
            }
            lib = mx_tree_next(lib);
        }
        fputs("</ul>", stream);
    }

    fputs("<h2>tags:</h2><ul>", stream);
    int n;
    const char *p, *text;

    fputs("<li><a href=\"function.htm\">function</a> Functions</li>", stream);
    fputs("<li><a href=\"type.htm\">type</a> Data types such as typedefs</li>", stream);
    fputs("<li><a href=\"define.htm\">define</a> Preprocessor defines</li>", stream);
    fputs("<li><a href=\"macro.htm\">macro</a> Preprocessor macros</li>", stream);
    fputs("<li><a href=\"global.htm\">global</a> Global variables</li>", stream);

    text = alltags.data;
    while (*text == ',')
        ++text;
    do {
        p = strchr(text, ',');
        n = (p) ? (p - text) : (int)strlen(text);

        if ((n) &&
            (strncmp("mainpage", text, n) != 0) &&
            (strncmp("library", text, n) != 0) &&
            (strncmp("module", text, n) != 0) &&
            (strncmp("function", text, n) != 0) &&
            (strncmp("type", text, n) != 0) &&
            (strncmp("define", text, n) != 0) &&
            (strncmp("macro", text, n) != 0) &&
            (strncmp("global", text, n) != 0) &&
            (strncmp("_top_", text, 5) != 0)) {
            STRING temp;

            mx_vector(&temp);
            mx_vector_append(&temp, text, n);

            fputs("<li><a href=\"", stream);
            sprint(&temp, stream, 0);
            fputs(".htm\">", stream);
            sprint(&temp, stream, 0);
            fputs("</a></li>", stream);

            mx_vector_free(&temp);
        }
        text = p + 1;
    } while ((p) && (*p) && (n));
    fputs("</ul>", stream);

    fputs("<h2>modules:</h2><ul>", stream);
    lib = mx_tree_first(&mainpage);
    while (lib) {
        DNODE* mod = mx_tree_first(lib);
        while (mod) {
            if (substantialmodule(mod)) {
                fputs("<li>", stream);
                fputs("<a href=\"", stream);
                if (mx_vector_size(&lib->name)) {
                    sprint(&lib->name, stream, 0);
                    fputs("_", stream);
                }
                sprint(&mod->name, stream, 0);
                fputs(".htm", stream);
                fputs("\">", stream);
                sprint(&mod->name, stream, 0);
                fputs("</a> \n", stream);
                writedocs(mod, stream, 0);
                fputs("</li>\n", stream);
            }
            mod = mx_tree_next(mod);
        }
        lib = mx_tree_next(lib);
    }
    fputs("</ul>", stream);
    
    fputs("<h2>documented objects:</h2><ul>", stream);
    lib = mx_tree_first(&mainpage);
    while (lib) {
        DNODE* mod = mx_tree_first(lib);
        while (mod) {
            DNODE* node = mx_tree_first(mod);
            while (node) {
                if (mx_vector_size(&node->name)) {
                    fputs("<li>", stream);
                    fputs("<a href=\"", stream);
                    if (mx_vector_size(&lib->name)) {
                        sprint(&lib->name, stream, 0);
                        fputs("_", stream);
                    }
                    sprint(&mod->name, stream, 0);
                    fputs(".htm", stream);
                    if (node->name.data[0] != '#')
                        fputs("#", stream);
                    sprint(&node->name, stream, 0);
                    fputs("\">", stream);
                    sprint(&node->name, stream, 0);
                    fputs("</a>\n", stream);
                    writedocs(node, stream, 0);
                    fputs("</li>\n", stream);
                }
                node = mx_tree_next(node);
            }
            mod = mx_tree_next(mod);
        }
        lib = mx_tree_next(lib);
    }
    fputs("</ul>", stream);
    
    htmlclose(stream);
    mx_vector_free(&title);
}

static void writelibraries(void)
{
    DNODE *lib = mx_tree_first(&mainpage);

    while (lib) {
        if (mx_vector_size(&lib->name) == 0)
            goto nextlib;
    
        FILE* stream = htmlopen(0, &lib->name, 1);
        assert(stream);

        writeheader(lib, stream, "library: ");
        fputs("<h2>modules:</h2><ul>", stream);

        DNODE* mod = mx_tree_first(lib);
        while (mod) {
            if (substantialmodule(mod)) {
                fputs("<li><a href=\"\n", stream);
                if (mx_vector_size(&lib->name)) {
                    sprint(&lib->name, stream, 0);
                    fputs("_", stream);
                }
                sprint(&mod->name, stream, 0);
                fputs(".htm\">", stream);
                sprint(&mod->name, stream, 0);
                fputs("</a> \n", stream);
                writedocs(mod, stream, 0);
                fputs("</li>\n", stream);
            }
            mod = mx_tree_next(mod);
        }
        fputs("</ul>", stream);

        fputs("<h2>documented objects:</h2><ul>", stream);
        mod = mx_tree_first(lib);
        while (mod) {
            DNODE* node = mx_tree_first(mod);
            while (node) {
                if (mx_vector_size(&node->name)) {
                    fputs("<li>", stream);
                    fputs("<a href=\"", stream);
                    if (mx_vector_size(&lib->name)) {
                        sprint(&lib->name, stream, 0);
                        fputs("_", stream);
                    }
                    sprint(&mod->name, stream, 0);
                    fputs(".htm", stream);
                    if (node->name.data[0] != '#')
                        fputs("#", stream);
                    sprint(&node->name, stream, 0);
                    fputs("\">", stream);
                    sprint(&node->name, stream, 0);
                    fputs("</a>\n", stream);
                    writedocs(node, stream, 0);
                    fputs("</li>\n", stream);
                }
                node = mx_tree_next(node);
            }
            mod = mx_tree_next(mod);
        }
        fputs("</ul>", stream);
        htmlclose(stream);
        
nextlib:
        lib = mx_tree_next(lib);
    }
}

static void writemodules(void)
{
    DNODE *lib = mx_tree_first(&mainpage);
    while (lib) {
        DNODE* mod = mx_tree_first(lib);
        while (mod) {
            if (substantialmodule(mod)) {
                FILE* stream = htmlopen(&lib->name, &mod->name, 1);
                assert(stream);

                writeheader(mod, stream, "module: ");

                if (mx_vector_size(&lib->name)) {
                    fputs("<p>Part of the <a href=\"", stream);
                    sprint(&lib->name, stream, 0);
                    fputs(".htm\">", stream);
                    sprint(&lib->name, stream, 0);
                    fputs("</a> library.", stream);
                }
                fputs("<ul>", stream);

                DNODE* node = mx_tree_first(mod);
            
                while (node) {
                   if (mx_vector_size(&node->name)) {
                       fputs("<li>\n", stream);
                       writenode(node, stream);
                       fputs("<p></li>\n", stream);
                   }
                   node = mx_tree_next(node);
                }
                fputs("</ul>", stream);

                htmlclose(stream);
            }
            mod = mx_tree_next(mod);
        }
        lib = mx_tree_next(lib);
    }
}

static void writetagheader(STRING* tag, FILE* stream)
{
    DNODE *lib = mx_tree_first(&mainpage);

    while (lib) {
        DNODE* mod = mx_tree_first(lib);
        while (mod) {
            DNODE* node = mx_tree_first(mod);
            while (node) {
                if ((matchtag(&node->tags, tag->data, mx_vector_size(tag))) &&
                    (matchtag(&node->tags, "_top_", 5))) {
                    writedocs(node, stream, 1);
                    fputs("<br>", stream);
                    writeref(lib, mod, stream);
                    fputs("<p>\n", stream);
                }
                node = mx_tree_next(node);
            }
            mod = mx_tree_next(mod);
        }
        lib = mx_tree_next(lib);
    }
}

static void writetagnodes(STRING* tag, FILE* stream)
{
    DNODE *lib = mx_tree_first(&mainpage);

    while (lib) {
        DNODE* mod = mx_tree_first(lib);
        while (mod) {
            DNODE* node = mx_tree_first(mod);
            while (node) {
                if ((matchtag(&node->tags, tag->data, mx_vector_size(tag))) &&
                    (!matchtag(&node->tags, "_top_", 5))) {
                    fputs("<li>\n", stream);
                    writenode(node, stream);
                    fputs("<br>", stream);
                    writeref(lib, mod, stream);
                    fputs("<p></li>\n", stream);
                }
                node = mx_tree_next(node);
            }
            mod = mx_tree_next(mod);
        }
        lib = mx_tree_next(lib);
    }
}

static void writetagpage(void)
{
    int n;
    const char * p, *text;

    FILE* stream;
    STRING title;

    mx_vector(&title);
    append_str(&title, "tags");
    
    stream = htmlopen(0, &title, 1);
    assert(stream);

    fputs("<h2>tags:</h2><ul>", stream);

    fputs("<li><a href=\"function.htm\">function</a> Functions</li>", stream);
    fputs("<li><a href=\"type.htm\">type</a> Data types such as typedefs</li>", stream);
    fputs("<li><a href=\"define.htm\">define</a> Preprocessor defines</li>", stream);
    fputs("<li><a href=\"macro.htm\">macro</a> Preprocessor macros</li>", stream);
    fputs("<li><a href=\"global.htm\">global</a> Global variables</li>", stream);

    text = alltags.data;
    while (*text == ',')
        ++text;
    do {
        p = strchr(text, ',');
        n = (p) ? (p - text) : (int)strlen(text);

        if ((n) &&
            (strncmp("mainpage", text, n) != 0) &&
            (strncmp("library", text, n) != 0) &&
            (strncmp("module", text, n) != 0) &&
            (strncmp("function", text, n) != 0) &&
            (strncmp("type", text, n) != 0) &&
            (strncmp("define", text, n) != 0) &&
            (strncmp("macro", text, n) != 0) &&
            (strncmp("global", text, n) != 0) &&
            (strncmp("_top_", text, 5) != 0)) {
            STRING temp;

            mx_vector(&temp);
            mx_vector_append(&temp, text, n);

            fputs("<li><a href=\"", stream);
            sprint(&temp, stream, 0);
            fputs(".htm\">", stream);
            sprint(&temp, stream, 0);
            fputs("</a></li>", stream);

            mx_vector_free(&temp);
        }
        text = p + 1;
    } while ((p) && (*p) && (n));
    fputs("</ul>", stream);

    htmlclose(stream);

    text = alltags.data;
    while (*text == ',')
        ++text;
    do {
        p = strchr(text, ',');
        n = (p) ? (p - text) : (int)strlen(text);

        if ((n) &&
            (strncmp("mainpage", text, n) != 0) &&
            (strncmp("library", text, n) != 0) &&
            (strncmp("module", text, n) != 0) &&
            (strncmp("_top_", text, 5) != 0)) {
            STRING temp;

            mx_vector(&temp);
            mx_vector_append(&temp, text, n);

            stream = htmlopen(0, &temp, 1);

            fputs("<h1>", stream);
            sprint(&temp, stream, 0);
            fputs("</h1>", stream);
            writetagheader(&temp, stream);
            fputs("<ul>", stream);
            writetagnodes(&temp, stream);
            fputs("</ul>", stream);

            htmlclose(stream);
            mx_vector_free(&temp);
        }

        text = p + 1;
    } while ((p) && (*p));
}

int main(int argc, char* argv[])
{
    char prepath[4096];

    printf("MXDOC 2.0\n"
           "Copyright (C) 2006 Douglas Eleveld\n"
           "MXDOC comes with ABSOLUTELY NO WARRANTY\n"
           "This is free software, and you are welcome to redistribute it\n"
           "under certain conditions;\n"
           "see http://www.deleveld.dds.nl/mxdoc/index.htm\n\n");

    mx_vector(&filename);
    mx_vector(&modname);
    mx_vector(&libname);
    mx_vector(&alltags);
    memset(&current, 0, sizeof(current));
    
    strcpy(prepath, ".");
    if (argc > 1) {
        strcat(prepath, "/");
        strcat(prepath, argv[1]);
    }

    find_modules(prepath);
    mx_vector_append1(&alltags, 0);

    writetagpage();
    writemodules();
    writelibraries();
    writemainpage();
    
    return EXIT_SUCCESS;
}

