/*
    MXDOC
    HTML documentation generator for C
    Copyright (C) 2006 Douglas Eleveld

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <dirent.h>
#include <sys/stat.h>

static void process_module(const char* d_name, char *text, const int len)
{
    struct stat sbuf;

    fprintf(stderr, "module: %s\n", d_name);
        
    /* Put in the module name */
    mx_vector_resize(&modname, 0);
    mx_vector_append(&modname, d_name, strlen(d_name) - 2);

    /* Process the header first */
    process_file(text + 2);

    /* Check for a corresponding c source file */
    text[len - 1] = 'c';
    if (stat(text, &sbuf) == 0)
        process_file(text + 2);

    /* Check for a corresponding module source subdirectory */
    text[len - 2] = 0;
    if ((stat(text, &sbuf) == 0) && (S_ISDIR(sbuf.st_mode))) {
        struct dirent* entry2;
        DIR * dir2 = opendir(text);

        /* Do C files within the module directory */
        while ((dir2 != 0) &&
               ((entry2 = readdir(dir2)) != 0)) {
            FILE * header;

            /* Skip traversal directories */
            if (entry2->d_name[0] == '.')
                continue;

            /* Only do module C files */
            if ((entry2->d_name[entry2->d_namlen - 2] != '.') ||
                (entry2->d_name[entry2->d_namlen - 1] != 'c'))
                continue;

            /* Skip C files with a corresponding header, these are sub-dirs
               orf a module and will be found when the dir is traversed when
               looking for headers/modules. */
            entry2->d_name[entry2->d_namlen - 1] = 'h';
            header = fopen(entry2->d_name, "r");
            if (header) {
                fclose(header);
                continue;
            }
            entry2->d_name[entry2->d_namlen - 1] = 'c';
                
            /* Process module C files */
            text[len - 2] = 0;
            strcat(text, "/");
            strcat(text, entry2->d_name);
            process_file(text + 2);
        }
        if (dir2)
            closedir(dir2);
    }
    mx_vector_resize(&modname, 0);
}

static void find_modules(const char *path)
{
    struct dirent* entry;
    DIR * dir = opendir(path);
    char *text = malloc(4096);

#ifdef __DJGPP__
    _djstat_flags |= _STAT_INODE | _STAT_EXEC_EXT | _STAT_EXEC_MAGIC | _STAT_DIRSIZE | _STAT_ROOT_TIME | _STAT_WRITEBIT;
#endif

    /* Look through all files in the directory */
    while ((dir) && ((entry = readdir(dir)) != 0)) {
        struct stat sbuf;

        /* Skip traversal directories */
        if (entry->d_name[0] == '.')
            continue;
        
        strcpy(text, path);
        strcat(text, "/");
        strcat(text, entry->d_name);
        int len = strlen(text);
        assert(len < 4000);

        if (stat(text, &sbuf) != 0)
            continue;

        /* Found a module header fle */
        if ((text[len - 2] == '.') && (text[len - 1] == 'h'))
            process_module(entry->d_name, text, len);
            
        /* Recurse sub-directories */
        if (S_ISDIR(sbuf.st_mode)) {

            /* Skip dirs with a corresponding header since these will be
            recursed when the header is found */
            int foundheader = 0;
            char* headername = malloc(len + 5);
            if (headername) {
                strcpy(headername, text);
                strcat(headername, ".h");

                if (stat(headername, &sbuf) == 0)
                    foundheader = 1;
                free(headername);
            }

            if (!foundheader) {
                STRING previous;

                /* Look for a library header */
                foundheader = 0;
                headername = malloc(len + entry->d_namlen + 5);
                if (headername) {
                    strcpy(headername, text);
                    strcat(headername, "/");
                    strcat(headername, entry->d_name);
                    strcat(headername, ".h");

                    if (stat(headername, &sbuf) == 0)
                        foundheader = 1;
                    free(headername);
                }

                /* There is a library header, so push library name */
                if (foundheader) {
                    mx_vector(&previous);
                    mx_vector_append(&previous, libname.data, mx_vector_size(&libname));

                    /* Push the new dir name */
                    mx_vector_resize(&libname, 0);
                    mx_vector_append(&libname, entry->d_name, strlen(entry->d_name));
                }

                /* Go in the sub dir looking for more modules */
                find_modules(text);

                /* Pop the old library name back */
                if (foundheader) {
                    mx_vector_resize(&libname, 0);
                    mx_vector_append(&libname, previous.data, mx_vector_size(&previous));
                    mx_vector_free(&previous);
                }
            }
        }
    }

    free(text);

    if (dir)
        closedir(dir);
}


