/** !A 100% typesafe generic vector for C.

This module contains macros, types and functions for MX_VECTOR whic is a
a 100% typesafe generic vector object for C.  The underlying type must be
bit-copy-correct which means that a binary copy retains proper functioning
and takes over ownership of any internal buffers.
<p>
The interface of MX_VECTOR is done through macros and generic type-indenpendent
functions perform the vector operations.  In contrast to C++ vector types,
the code size when using MX_VECTOR is constant regardless of how many vector
types are instantiated.
<p>
The correct operation of MX_VECTOR depends on some things that not in the C
standard:<ul>
<li>Size of pointer to void and pointer to other types are the same size,
representation and alignment</li>
<li>Writing data in a element of a union and reading out of a different
element works as expected</li>
</ul>
All of the compilers I have tested give no warnings on the
MX_VECTOR code and seem to function properly.  I have tested 16 and 32 bit
compilers as well as compilers with various byte orders.
<p>
MX_VECTOR objects themselves are bit-copy-correct which means that memory 
containing a MX_VECTOR can be safely realloc'd and MX_VECTOR can be used 
inside another MXVECTOR.
<p>
Here is a simple example for doubles but the same concept applies to all
bit-copy-correct types.
<pre>
size_t i;
MX_VECTOR(double) dvect;

mx_vector(&dvect);
mx_vector_resize(&dvect, 100);

for (i=0; i &lt; mx_vector_size(&dvect); i++)
   dvect.data[i] = i;
   
mx_vector_free(&dvect);
</pre>

NOTE: All of the mx_vector* functions are in fact macros that may evaluate
the arguments (the vector) more than once.  This is especially true for the
vector itself (the first argument) and any pointer arguments.  Efforts
have been made to evaluate other arguments only once.  So modifiying
arguments in the macro like this:
<pre>
mx_vector_resize(&dvect, n++);
</pre>
will work as expected.  However code like this:
<pre>
mx_vector_resize(vectptr++, n);
</pre>
will not work as expected.
*/

#ifndef MX__VECTOR_HEADER
#define MX__VECTOR_HEADER

#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

/* Internal vector representation */
    typedef struct {
	void *_data;
	size_t _size;
	size_t _capacity;
    } MX__VECTOR_DATA;

/**type !A 100% typesafe generic vector for C
This macro functions as a type delaration for a vector of user-defined
type. */
#define MX_VECTOR(type)               \
    union {                           \
        MX__VECTOR_DATA _vector;      \
        type* data;                   \
    }

/**type !An iterator for a MX_VECTOR
This macro functions as a type declaration for an iterator for a
similarly typed MX_VECTOR. */
#define MX_VECTOR_ITERATOR(type) \
    type*

/**func !Construct a MX_VECTOR
This macro constructs a MX_VECTOR object.  This must be called first
before any other mx_vector_* functions are called.
<p>MX_VECTOR is correctly initialized with a 0 bit pattern.  So static 
MX_VECTOR are correctly initialized and dont necessarily need a call to the 
constructor. */
#define mx_vector(mxvect) \
    do { \
       assert(mxvect); \
       mx__vector(&(mxvect)->_vector); \
    } while (0)

/**func !Destruct a MX_VECTOR
This macro destroys an MX_VECTOR object by releasing any memory
that has been allocated.  It should be the last call to any mx_vector_* 
functions.  Access of the MX_VECTOR object after destruction should not be done. */
#define mx_vector_free(mxvect) \
    do { \
        assert(mxvect); \
        mx__vector_free(&(mxvect)->_vector); \
    } while (0)

/**func !Return an element from a MX_VECTOR at a specific index
This macro returns elements from a MX_VECTOR at a specific index.
It is safer than accessing the underlying pointer because it wont allow
access outside the correct range for the vector.  */
#define mx_vector_at(mxvect, mxn) \
    (mx__vector_rangecheck(&(mxvect)->_vector, (mxn)), (mxvect)->data[(mxn)])

/**func !Return the last element from a MX_VECTOR
This macro returns the last element from a MX_VECTOR. */
#define mx_vector_back(mxvect) \
    (mx__vector_rangecheck(&(mxvect)->_vector, 0), (mxvect)->data[(mxvect)->_vector._size - 1])

/**func !Pointer to first element
This macro returns a pointer to the first data element in the MX_VECTOR object. 
If the MX_VECTOR contains no data elements then 0 is returned. */
#define mx_vector_begin(mxvect) \
    ((((mxvect)) && ((mxvect)->data) && ((mxvect)->_vector._size)) ? (&(mxvect)->data[0]) : (0));
    
/**func !Capacity
This macro returns the capacity of an MX_VECTOR object.  This is the 
number of elements that the vector can contain without reqriring any internal
memory management. */
#define mx_vector_capacity(mxvect) \
    ((const size_t)((mxvect)->_vector._capacity))

/**func !Clear
This macro deletes all the objects from a MX_VECTOR */
#define mx_vector_clear(mxvect) \
    do { \
        (mxvect)->_vector.size = 0; \
    while (0)

/**func !Empty
This macro deletes all the objects from a MX_VECTOR */
#define mx_vector_empty(mxvect) \
    (((mxvect)->_vector.size == 0) ? (1) : (0))

/** !Pointer one-past last element
This macro returns a pointer one-past the last data element in the MX_VECTOR object. 
If the MX_VECTOR contains no data elements then 0 is returned. */
#define mx_vector_end(mxvect) \
    ((((mxvect)) && ((mxvect)->data) && ((mxvect)->_vector._size)) ? (&(mxvect)->data[(mxvect)->_vector._size]) : (0));

/** !Erase an element
This macro removes a data element from an MX_VECTOR object.
If there are data elements at indicies higher than those removed then they
are are moved to lower indicies.  */
#define mx_vector_erase(mxvect, mxiter) \
    mx__vector_remove(&(mxvect)->_vector, ((mxiter) - (mxvect)->data), (1), sizeof(*(mxvect)->data));

/** !Push an element onto the back of the MX_VECTOR
This macro makes a bitwise copy of a single data element and appends 
the copy onto an existing MX_VECTOR object. The size of the vector is expanded 
to hold the new data elements. */
#define mx_vector_push_back(mxvect, mxdat) \
    do { \
        const size_t mx__vector_index = mx__vector_makeroom(&(mxvect)->_vector, 1, (mxvect)->_vector._size, sizeof(*(mxvect)->data)); \
        (mxvect)->data[mx__vector_index] = (mxdat); \
    } while (0) \





#define mx_vector_remove(mxvect, ind, n) mx__vector_remove(&(mxvect)->_vector, (ind), (n), sizeof(*(mxvect)->data))

/** !Number of elements
This macro returns the size of an MX_VECTOR object.  This is the number 
of data elements that can be safely accessed.  */
#define mx_vector_size(mxvect) \
    ((const size_t)((mxvect)->_vector._size))

/** !Resize
This macro resizes a MX_VECTOR object.  If the MX_VECTOR does not have 
sufficient capacity to hold the new number of elements then new space is 
reserved using mx_vector_reserve(). */
#define mx_vector_resize(mxvect, n) \
    mx__vector_resize(&(mxvect)->_vector, (n), sizeof(*(mxvect)->data))

/** !Set capacity
This macro reserves space in a MX_VECTOR object for later copying or
appending of data elements.  If enough space is reserved the insertion or appending
of elements does not require memory management (realloc of internal buffer) and 
will be fast.  If enough space is already reserved this macro does nothing.  */
#define mx_vector_reserve(mxvect, n) \
    mx__vector_reserve(&(mxvect)->_vector, (n), sizeof(*(mxvect)->data))

/** !Increase capacity
This macro reserves space for additional data elements in a MX_VECTOR object.
If enough space is reserved then future insertion or appending may not require memory
management (realloc of internal buffer) and will therefore be fast. If enough space
is already reserved this macro does nothing.*/
#define mx_vector_extend(mxvect, n) \
    mx_vector_reserve((mxvect), (mxvect)->_vector._size + (n))

/** !Reduce capacity to minimum
This macro reduces the memory used by a MX_VECTOR object to the 
minimum required to safely contain the data elements contained. This function can 
cause memory management (realloc of internal buffer) to occur. */
#define mx_vector_contract(mxvect) \
    mx__vector_contract(&(mxvect)->_vector,sizeof(*(mxvect)->data))

/** !Append 1 element (bitwise copy)
This macro makes a bitwise copy of a single data element and appends 
the copy onto an existing MX_VECTOR object. The size of the vector is expanded 
to hold the new data elements. */
#define mx_vector_append1(mxvect, dat) \
    do { \
        const size_t mx__vector_index = mx__vector_makeroom(&(mxvect)->_vector, 1, (mxvect)->_vector._size, sizeof(*(mxvect)->data)); \
        (mxvect)->data[mx__vector_index] = (dat); \
    } while (0) \

/** !Append a number of elements (bitwise copy)
This macro makes a bitwise copy of a number of data elements and appends the copies 
onto an existing MX_VECTOR object.  More than one data element can be appended at once. 
If necessary the size of the vector is expanded to hold the new data elements. */
#define mx_vector_append(mxvect, dat, n) \
    do { \
        size_t mx__vector_i; \
        const size_t mx__n = (n); \
        const size_t mx__vector_index = mx__vector_makeroom(&(mxvect)->_vector, (n), (mxvect)->_vector._size, sizeof(*(mxvect)->data)); \
        for (mx__vector_i=0; mx__vector_i<mx__n; ++mx__vector_i) \
            (mxvect)->data[mx__vector_index + mx__vector_i] = (dat)[mx__vector_i]; \
    } while (0)

/** !Insert 1 element (bitwise copy)
This macro makes a bitwise copy of one data element and inserts the
copy into an existing MX_VECTOR object.  If necessary the size of the vector 
is expanded to hold the new data element. */
#define mx_vector_insert1(mxvect, dat, ind) \
    do { \
        const size_t mx__vector_index = mx__vector_makeroom(&(mxvect)->_vector, 1, (ind), sizeof(*(mxvect)->data)); \
        (mxvect)->data[mx__vector_index] = *(dat); \
    } while (0) \

/** !Insert a number of elements (bitwise copy)
This macro makes a bitwise copy of a number of data elements and inserts the
copies into an existing MX_VECTOR object at sepcific indicies.  More than one data 
element can be inserted at once.  If necessary the size of the vector is expanded 
to hold the new data elements. */
#define mx_vector_insert(mxvect, dat, n, ind) \
    do { \
        size_t mx__vector_i; \
        const size_t mx__n = (n); \
        const size_t mx__vector_index = mx__vector_makeroom(&(mxvect)->_vector, (mx__n), (ind), sizeof(*(mxvect)->data)); \
        for (mx__vector_i=0; mx__vector_i<mx__n; ++mx__vector_i) \
            (mxvect)->data[mx__vector_index + mx__vector_i] = (dat)[mx__vector_i]; \
    } while (0)

/** !Remove last element
This macro removes only the last object from a vector.  The size of the MX_VECTOR 
is reduced by one. */
#define mx_vector_remove_last(mxvect) \
    mx__vector_remove(&(mxvect)->_vector, (mxvect)->_vector._size-1, 1, sizeof(*(mxvect)->data))

    void mx__vector(MX__VECTOR_DATA * mxvect);
    void mx__vector_free(MX__VECTOR_DATA * mxvect);

    void mx__vector_resize(MX__VECTOR_DATA * mxvect,
			   const size_t newsize,
			   const size_t sizeofdata);
    void mx__vector_reserve(MX__VECTOR_DATA * mxvect,
			    const size_t newsize,
			    const size_t sizeofdata);
    void mx__vector_contract(MX__VECTOR_DATA * mxvect,
			     const size_t sizeofdata);

    void mx__vector_remove(MX__VECTOR_DATA * mxvect,
			   const size_t index, size_t num,
			   const size_t sizeofdata);
    size_t mx__vector_makeroom(MX__VECTOR_DATA * ptr,
				     const size_t num,
				     const size_t place,
				     const size_t sizeofdata);
    void mx__vector_rangecheck(MX__VECTOR_DATA * vect, const size_t n);
                     

#ifdef __cplusplus
}
#endif
#endif
/*
    DEDS
    Data structures and utility functions for C
    Copyright (C) 2006 Douglas Eleveld

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
