#include "depui/depui.h"

#ifdef MX_PLATFORM_MGRX

#   define MXMODULE_DRS

#   define GRX_SKIP_INLINES
#   include <mgrx.h>
#   include <mgrxkeys.h>

static MX_RECT mx__mgrx_screen;
static GrEvent mx__grx_event;
static GrTextOption mx__textoption;
static GrColor *mx__egacolors = 0;
static int mx__grx_pressed = 0;
static int mx__grx_init = 0;

static GrContext buffer;
static int bufferx, buffery;

#   define bufferw 256
#   define bufferh 192

#   define BLACK        mx__egacolors[0]
#   define BLUE         mx__egacolors[1]
#   define GREEN        mx__egacolors[2]
#   define CYAN         mx__egacolors[3]
#   define RED          mx__egacolors[4]
#   define MAGENTA      mx__egacolors[5]
#   define BROWN        mx__egacolors[6]
#   define LIGHTGRAY    mx__egacolors[7]
#   define DARKGRAY     mx__egacolors[8]
#   define LIGHTBLUE    mx__egacolors[9]
#   define LIGHTGREEN   mx__egacolors[10]
#   define LIGHTCYAN    mx__egacolors[11]
#   define LIGHTRED     mx__egacolors[12]
#   define LIGHTMAGENTA mx__egacolors[13]
#   define YELLOW       mx__egacolors[14]
#   define WHITE        mx__egacolors[15]

static MX_PLATFORM_FONT mx__mgrx_font[] = {
	{&GrFont_PC6x8, "Builtin 6x8"},
	{&GrFont_PC8x8, "Builtin 8x8"},
	{&GrFont_PC8x14, "Builtin 8x14"},
	{&GrFont_PC8x16, "Builtin 8x16"},
};

#   define MX__ARRAY(a) (sizeof(a)/sizeof(a[0]))

const MX_PLATFORM_FONT *mx_platform_font_system(const int i)
{
	if (i < 0)
		return &mx__mgrx_font[1];

	if (i < (int) MX__ARRAY(mx__mgrx_font))
		return &mx__mgrx_font[i];

	return 0;
}

void *mx_platform_font_load(const char *filename)
{
	return GrLoadFont((char *) filename);
}

const char *mx_platform_font_path(void)
{
	return "*.fnt";
}

void mx_platform_font_free(void *font)
{
	GrFont *f = (GrFont *) font;

	GrUnloadFont(f);
}

unsigned int mx_platform_font_height(void *font)
{
	if (!font)
		font = &GrDefaultFont;

	mx__textoption.txo_font = (GrFont *) font;
	mx__textoption.txo_fgcolor.v = BLACK;

	return GrStringHeight("HI", 2, &mx__textoption);
}

unsigned int mx_platform_font_width(void *font, const char *text, int len)
{
	if (!font)
		font = &GrDefaultFont;

	mx__textoption.txo_font = (GrFont *) font;
	mx__textoption.txo_fgcolor.v = BLACK;

	if (len < 0)
		len = strlen(text);

	return GrStringWidth((void *) text, len, &mx__textoption);
}

static const MX_PLATFORM_RESOLUTION mx__mgrx_resolutions[] = {
	{640, 480, "640x480"},
	{800, 600, "800x600"},
	{1024, 768, "1024x768"},
	{1280, 1024, "1280x1024"},
	{0, 0, 0}
};

static const MX_PLATFORM_DEPTH mx__mgrx_depths[] = {
	{4, "4"},
	{8, "8"},
	{16, "16"},
	{24, "24"},
	{32, "32"},
	{0, 0}
};

static const MX_PLATFORM_DRIVER mx__mgrx_drivers[] = {
	{0, 0}
};

extern const MX_THEME mx_theme_mgrx;

static MX_PLATFORM_THEME mx__mgrx_themes[] = {
	{&mx_theme_mgrx, "mgrx theme"},
	{0, 0},
};

void mx_platform_modes(const MX_PLATFORM_DRIVER * d[], const MX_PLATFORM_RESOLUTION * r[], const MX_PLATFORM_DEPTH * c[],
					   const MX_PLATFORM_THEME * t[])
{
	*d = mx__mgrx_drivers;
	*r = mx__mgrx_resolutions;
	*c = mx__mgrx_depths;
	*t = mx__mgrx_themes;
}

unsigned mx_platform_start(int w, int h, int c, const void *driver)
{
	(void) driver;

	if (w == 0)
		w = 640;
	if (h == 0)
		h = 480;
	if (c == 0)
		c = 16;

	if (mx__grx_init) {
		GrMouseEraseCursor();
		GrEventUnInit();
	}

	if (!GrSetMode(GR_width_height_bpp_graphics, w, h, c))
		return false;

	if (!GrMouseDetect())
		return false;

	GrEventInit();
	GrEventGenMmove(GR_GEN_MMOVE_ALWAYS);
	mx__grx_init = 1;

	GrMouseDisplayCursor();

	mx__mgrx_screen.x1 = 0;
	mx__mgrx_screen.y1 = 0;
	mx__mgrx_screen.x2 = GrMaxX();
	mx__mgrx_screen.y2 = GrMaxY();

	mx_drs_area(mx__mgrx_screen.x2, mx__mgrx_screen.y2);

	return true;
}

void mx_platform_stop(void)
{
	GrEventUnInit();
	GrMouseEraseCursor();
	GrSetMode(GR_default_text);

	mx__grx_init = 0;
}

unsigned mx_platform_clip(const MX_RECT * r)
{
	GrSetClipBox(bufferx + r->x1, buffery + r->y1, bufferx + r->x2, buffery + r->y2);
	return true;
}

const MX_RECT *mx_platform_rect(void)
{
	mx__mgrx_screen.x2 = GrMaxX();
	mx__mgrx_screen.y2 = GrMaxY();

	return &mx__mgrx_screen;
}

static void mx__mgrx_flush(MX_RECT * rrect)
{
	MX_RECT brect;

	GrSetContext(&buffer);
	brect.x1 = rrect->x1;
	brect.y1 = rrect->y1;
	brect.x2 = rrect->x1 + GrMaxX();
	brect.y2 = rrect->y1 + GrMaxY();

	MXRECT_INTERSECT(brect, *rrect, *rrect);

	bufferx = -rrect->x1;
	buffery = -rrect->y1;
	mx__gui_redraw(rrect);

	GrSetContext(NULL);
	GrMouseEraseCursor();

	GrBitBlt(NULL, -bufferx, -buffery, &buffer, 0, 0, rrect->x2 - rrect->x1, rrect->y2 - rrect->y1, GrWRITE);

	GrMouseDisplayCursor();
}

unsigned mx_platform_poll(void)
{
	GrEventRead(&mx__grx_event);

	mx_drs_update(mx__mgrx_flush);

	return true;
}

void mx_platform_dirty(const MX_RECT * rect)
{
	mx_drs_dirty(rect, true);
}

unsigned mx_platform_pointer(int *px, int *py, int *pb)
{
	switch (mx__grx_event.type) {

	case GREV_MOUSE:
		*px = mx__grx_event.p2;
		*py = mx__grx_event.p3;

		if (mx__grx_event.p1 == GRMOUSE_LB_PRESSED)
			mx__grx_pressed = 1;
		else if (mx__grx_event.p1 == GRMOUSE_LB_RELEASED)
			mx__grx_pressed = 0;
		*pb = mx__grx_pressed;

		break;

	case GREV_MMOVE:
		*px = mx__grx_event.p2;
		*py = mx__grx_event.p3;
		mx__grx_pressed = (mx__grx_event.p1 == 1) ? 1 : 0;
		*pb = mx__grx_pressed;
		break;

	default:
		return false;
	}
	mx__grx_event.type = GREV_NULL;
	return true;
}

unsigned mx_platform_key(int *scan, int *ascii)
{
	if (mx__grx_event.type != GREV_KEY)
		return false;

	*scan = 0;
	*ascii = 0;

	if (mx__grx_event.p1 & 0xff00)
		*scan = mx__grx_event.p1;
	else
		*ascii = mx__grx_event.p1;

	mx__grx_event.type = GREV_NULL;
	return true;
}

static unsigned mx_theme_mgrx_start(void)
{
	GrCreateContext(bufferw, bufferh, NULL, &buffer);

	mx__egacolors = GrAllocEgaColors();
	mx__textoption.txo_font = &GrDefaultFont;
	mx__textoption.txo_fgcolor.v = BLACK;
	mx__textoption.txo_bgcolor.v = GrNOCOLOR;
	mx__textoption.txo_chrtype = GR_BYTE_TEXT;
	mx__textoption.txo_direct = GR_TEXT_RIGHT;
	mx__textoption.txo_xalign = GR_ALIGN_LEFT;
	mx__textoption.txo_yalign = GR_ALIGN_TOP;

	return true;
}

static void mx_theme_mgrx_stop(void)
{
	GrDestroyContext(&buffer);
}

static void mx_theme_mgrx_event(void)
{
	mx__event_default();

	if ((mx.event == MX_DEFAULTRECT) && (MXID(mx.obj) == MXID_WINRESIZE)) {
		MX_RECT *rect = mx_defaultrect_data();

		rect->y2 = rect->y1 + (rect->y2 - rect->y1) / 2;
	}
}

static void mx__mgrx_frame(const MX_RECT * objrect, int edge, int fill)
{
	GrHLine(bufferx + objrect->x1, bufferx + objrect->x2, buffery + objrect->y1, edge);
	GrHLine(bufferx + objrect->x1, bufferx + objrect->x2, buffery + objrect->y2, edge);
	GrVLine(bufferx + objrect->x1, buffery + objrect->y1 + 1, buffery + objrect->y2 - 1, edge);
	GrVLine(bufferx + objrect->x2, buffery + objrect->y1 + 1, buffery + objrect->y2 - 1, edge);

	GrFilledBox(bufferx + objrect->x1 + 1, buffery + objrect->y1 + 1, bufferx + objrect->x2 - 1, buffery + objrect->y2 - 1, fill);
}

static void mx_theme_mgrx_obj(MX_OBJ * obj)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(obj);

		mx__mgrx_frame(objrect, WHITE, LIGHTGRAY);
	}
}

static void mx_theme_mgrx_vslider(MX_SLIDER * slider)
{
	if (mx_exposing()) {
		int fill = WHITE;
		MX_RECT objrect = *MXRECT(slider);

		mx__mgrx_frame(&objrect, DARKGRAY, LIGHTGRAY);

		if (mx_armed(slider))
			fill = GREEN;

		objrect.y2 = objrect.y1 + slider->base.slider._lower;
		objrect.y1 += slider->base.slider._upper;

		mx__mgrx_frame(&objrect, BLACK, fill);
	}
}

static void mx_theme_mgrx_hslider(MX_SLIDER * slider)
{
	if (mx_exposing()) {
		int fill = WHITE;
		MX_RECT objrect = *MXRECT(slider);

		mx__mgrx_frame(&objrect, DARKGRAY, LIGHTGRAY);

		if (mx_armed(slider))
			fill = GREEN;

		objrect.x2 = objrect.x1 + slider->base.slider._lower;
		objrect.x1 += slider->base.slider._upper;

		mx__mgrx_frame(&objrect, BLACK, fill);
	}
}

static void mx__textchunks(const MX_TEXTUAL_DATA * textual, int fore, int x, int y)
{
	MX__TEXTCHUNK *chunk = mx_dllist_first(textual);
	void *font = mx__textual_platform_font(textual);

	const int x1 = mx_x1(textual);
	const int y1 = mx_y1(textual);

	/* Left and top alignment is correct with offset 0 */
	int offx = 0, offy = 0;

	if (textual->_align & MX_ALIGN_RIGHT)
		offx = mx_w(textual) - textual->_textwidth - 2 * x;
	if (textual->_align & MX_ALIGN_HCENTER)
		offx = (mx_w(textual) - textual->_textwidth - 2 * x) / 2;

	if (textual->_align & MX_ALIGN_BOTTOM)
		offy = mx_h(textual) - textual->_textheight - 2 * y;
	if (textual->_align & MX_ALIGN_VCENTER)
		offy = (mx_h(textual) - textual->_textheight - 2 * y) / 2;

	if (!font)
		font = &GrDefaultFont;

	while (chunk) {
		int len = 0;
		const char *text = mx_string_text(&chunk->text, &len);

		if (chunk->cursor) {
			GrColor fill = LIGHTGRAY;

			if (mx_focused(textual))
				fill = GREEN;

			GrFilledBox(bufferx + x + x1 + chunk->rect.x1, buffery + y + y1 + chunk->rect.y1, bufferx + x + x1 + chunk->rect.x2,
						buffery + y + y1 + chunk->rect.y2, fill);
		}

		if (text) {
			if (len < 0)
				len = strlen(text);

			mx__textoption.txo_font = (GrFont *) font;
			mx__textoption.txo_fgcolor.v = fore;

			GrDrawString((char *) text, len, bufferx + x + x1 + chunk->rect.x1 + offx, buffery + y + y1 + chunk->rect.y1 + offy, &mx__textoption);
		}

		chunk = mx_dllist_next(chunk);
	}
}

static int mx__textback(const MX_TEXTUAL_DATA * textual, int x1, int y1, int x2, int y2, unsigned armable)
{
	int fore = BLACK;
	int back = WHITE;

	if (mx_selected(textual)) {
		back = GREEN;
		fore = WHITE;
	}

	if (mx_disabled(textual))
		fore = DARKGRAY;

	if ((armable) && (mx_armed(textual)))
		back = LIGHTGRAY;

	GrFilledBox(bufferx + x1, buffery + y1, bufferx + x2, buffery + y2, back);

	return fore;
}

static void mx_theme_mgrx_textual(MX_TEXTUAL * text)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(text);
		int fore = mx__textback(MXTEXTUAL(text), objrect->x1, objrect->y1,
								objrect->x2, objrect->y2, false);

		mx__textchunks(MXTEXTUAL(text), fore, 0, 0);
	}
}

static void mx_theme_mgrx_scrollcorner(MX_OBJ * scrollcorner)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(scrollcorner);

		GrFilledBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, LIGHTGRAY);
	}
}

static void mx_theme_mgrx_scroll(MX_SCROLL * scroll)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(scroll);

		GrFilledBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, WHITE);
	}
}

static void mx_theme_mgrx_button(MX_BUTTON * button)
{
	if (mx_exposing()) {
		int fore;
		const MX_RECT *objrect = MXRECT(button);
		int offset = 0;

		if (mx_selected(button))
			offset = 1;

		if (mx_armed(button))
			GrBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, RED);

		else if (mx_selected(button))
			GrBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, WHITE);

		else
			GrBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, DARKGRAY);

		fore = mx__textback(MXTEXTUAL(button), objrect->x1 + 1, objrect->y1 + 1, objrect->x2 - 1, objrect->y2 - 1, true);

		mx__textchunks(MXTEXTUAL(button), fore, 2 + offset, 2 + offset);
	}
}

void mx_theme_mgrx_listelem(MX_LISTELEM * listelem)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(listelem);

		int fore = mx__textback(MXTEXTUAL(listelem), objrect->x1, objrect->y1,
								objrect->x2, objrect->y2, true);

		mx__textchunks(MXTEXTUAL(listelem), fore, 0, 0);
	}
}

static void mx_theme_mgrx_root(MX_WIN * root)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(root);

		GrFilledBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, LIGHTGRAY);
	}
}

static void mx_theme_mgrx_winborder(MX_WINBORDER * border)
{
	if (mx_exposing()) {
		MX_WIN *owner = MXOBJ(border)->_win;
		const MX_RECT *objrect = MXRECT(border);
		int fore = BLACK;
		int fill = WHITE;

		if (mx_active(owner))
			fill = GREEN;

		if (mx_disabled(owner))
			fore = LIGHTGRAY;

		mx__mgrx_frame(objrect, BLACK, fill);

		if (owner)
			mx__textchunks(MXTEXTUAL(owner), fore, mx_x(border) - mx_x(owner) + 2, mx_y(border) - mx_y(owner) + 2);
	}
}

static void mx_theme_mgrx_win(MX_WIN * win)
{
	if (mx_exposing()) {
		const MX_RECT *objrect = MXRECT(win);

		GrFilledBox(bufferx + objrect->x1, buffery + objrect->y1, bufferx + objrect->x2, buffery + objrect->y2, LIGHTGRAY);
	}
}

const MX_THEME mx_theme_mgrx = {
	mx_theme_mgrx_start,
	mx_theme_mgrx_stop,
	mx_theme_mgrx_event,
	mx_theme_mgrx_obj,
	mx_theme_mgrx_vslider,
	mx_theme_mgrx_hslider,
	mx_theme_mgrx_textual,
	mx_theme_mgrx_scrollcorner,
	mx_theme_mgrx_scroll,
	mx_theme_mgrx_button,
	mx_theme_mgrx_listelem,
	mx_theme_mgrx_root,
	mx_theme_mgrx_winborder,
	mx_theme_mgrx_win
};

#endif
