/** Graphics mode and driver information

This module contains DEGFX graphics mode and driver information. */
#ifndef MX__DEGFX_DRIVER_HEADER
#define MX__DEGFX_DRIVER_HEADER

#include "detk/bitfield.h"
#include "detk/drs.h"
#include "detk/region.h"
#include "degfx/types.h"
#include "degfx/colors.h"
#include "degfx/bitmap.h"

#ifdef __cplusplus
extern "C" {
#endif

    /** !Graphic driver information
    The basic graphic driver arguments and parameters.
    All elements can be replaced with 0 and a default choice will be made. */
        typedef struct MX_GFX_ARGS {
        /** The graphics width, height and color depth in bits. */
                int w, h, c;

                /** The title of the graphics window (for windowed modes) */
                const char *title;

                /** The buffer used for redrawing.  Larger buffers mean more 
                efficient redraws at the cost of more memory. */
                struct MX_BITMAP *buffer;

                /** The requested driver. */
                const struct MX_DRIVER *driver;

                /** The redraw function. */
                MX_REDRAW_FUNC redraw;

                /** The area of the graphics screen. */
                MX_RECT screen;

                /** The session counter is incremented every time the graphics
                mode is changed.  Keeping track of the session number you can
                check if the graphics mode has been changed. */
                unsigned int session;

                /** The pointer field is non-zero if the platform draws it's 
                own mouse.  */
         MX_BITFIELD(pointer);

                 MX_BITFIELD(_hidepointer);
        } MX_GFX_ARGS;

        typedef struct MX_DRIVER {
                unsigned (*_start) (MX_GFX_ARGS * args);
                void (*_stop) (void);

                unsigned (*_poll) (void);
                void (*_dirty) (const MX_RECT * rect);

                unsigned (*_pointer) (void);
                unsigned (*_key) (int *scan, int *ascii);
        } MX_DRIVER;

        typedef void (*MX_DRAW_FUNC) (const MX_RECT * rect, MX_PIXEL * array,
                                                                  const int stride);

        extern MX_GFX_ARGS mx__args;
        extern int mx__mousex, mx__mousey, mx__mouseb;

        extern void mx__gfx_flush(MX_DRAW_FUNC draw, MX_RECT * rect);
        extern void mx__gfx_update(MX_DRAW_FUNC draw, const MX_RECT * rect);

/* Drivers for DJGPP on DOS */
#if defined(__DJGPP__)
#define MX__DRIVER_DJGPP

/* Drivers for TinyCC (currently assuming win32) */
#elif defined(__TINYC__)
#define MX__DRIVER_WIN32_GDI

/* Drivers for Digital Mars (currently assuming win32) */
#elif defined(__DMC__)
#define MX__DRIVER_WIN32_GDI

/* Drivers for MINGW32 on win32 */
#elif defined(__WIN32__) && defined(__MINGW32__)
#define MX__DRIVER_WIN32_GDI

/* Drivers for Borlands free compiler */
#elif defined(__WIN32__) && (__BORLANDC__ >= 0x550)
#define MX__DRIVER_WIN32_GDI

/* Drivers for LCC on win32 */
#elif defined(__LCC__)
#define MX__DRIVER_WIN32_GDI

/* Drivers for Open Watcom for windows */
#elif defined(_WIN32) && defined(__WATCOMC__)
#define MX__DRIVER_WIN32_GDI

/* Drivers for Linux and Svgalib */
#elif defined(__linux__)
#define MX__DRIVER_SVGALIB

/* Drivers for QDOS on Sinclair QL */
#elif defined(__QDOS__)
#define MX__DRIVER_QDOS

/* Drivers for DOS and TURBOC */
#elif defined(__MSDOS__) && defined(__TURBOC__)
#define MX__DRIVER_TURBOC

/* If we dont recognize the platform, use empty driver */
#else
#define MX__DRIVER_EMPTY
#endif

#if defined(MX__DRIVER_DJGPP)
        extern const MX_DRIVER mx__djgpp_vesa;
        extern const MX_DRIVER mx__djgpp_vga;
        extern const MX_DRIVER mx__djgpp_13h;

/** The fullscreen driver. */
#define mx_driver_fullscreen mx__djgpp_vesa

/** The windowed driver. */
#define mx_driver_windowed mx__djgpp_vesa

/** The vga driver. */
#define mx_driver_vga mx__djgpp_vga

/** The mode 13h driver. */
#define mx_driver_13h mx__djgpp_13h

#elif defined(MX__DRIVER_WIN32_GDI)
        extern const MX_DRIVER mx__win32_gdi;
#define mx_driver_fullscreen mx__win32_gdi
#define mx_driver_windowed mx__win32_gdi
#define mx_driver_vga mx__win32_gdi
#define mx_driver_13h mx__win32_gdi

#elif defined(MX__DRIVER_SVGALIB)
        extern const MX_DRIVER mx__svgalib;
#define mx_driver_fullscreen mx__svgalib
#define mx_driver_windowed mx__svgalib
#define mx_driver_vga mx__svgalib
#define mx_driver_13h mx__svgalib

#elif defined(MX__DRIVER_QDOS)
        extern const MX_DRIVER mx__qdos_gfxlib;
#define mx_driver_fullscreen mx__qdos_gfxlib
#define mx_driver_windowed mx__qdos_gfxlib
#define mx_driver_vga mx__qdos_gfxlib
#define mx_driver_13h mx__qdos_gfxlib

#elif defined(MX__DRIVER_TURBOC)
        extern const MX_DRIVER mx__turboc;
#define mx_driver_fullscreen mx__turboc
#define mx_driver_windowed mx__turboc
#define mx_driver_vga mx__turboc
#define mx_driver_13h mx__turboc

#elif defined(MX__DRIVER_EMPTY)
        extern const MX_DRIVER mx__empty;
#define mx_driver_fullscreen mx__empty
#define mx_driver_windowed mx__empty
#define mx_driver_vga mx__empty
#define mx_driver_13h mx__empty
#endif

/* Driver modules */
#if defined(MXMODULE_DRIVER_ALL) || !defined(MX__DEGFX_SOURCE_LIBRARY)
#define MXMODULE_DRIVER_FULLSCREEN
#define MXMODULE_DRIVER_WINDOWED
#define MXMODULE_DRIVER_VGA
#define MXMODULE_DRIVER_13H
#endif

#if !defined(MXMODULE_DRIVER_FULLSCREEN) && \
    !defined(MXMODULE_DRIVER_WINDOWED) && \
    !defined(MXMODULE_DRIVER_VGA) && \
    !defined(MXMODULE_DRIVER_13H)
#define MXMODULE_DRIVER_FULLSCREEN
#endif

#ifdef MXMODULE_DRIVER_FULLSCREEN
#ifndef MX__DRIVER_DEFAULT
#define MX__DRIVER_DEFAULT mx_driver_fullscreen
#endif
#endif

#ifdef MXMODULE_DRIVER_WINDOWED
#ifndef MX__DRIVER_DEFAULT
#define MX__DRIVER_DEFAULT mx_driver_windowed
#endif
#endif

#ifdef MXMODULE_DRIVER_VGA
#ifndef MX__DRIVER_DEFAULT
#define MX__DRIVER_DEFAULT mx_driver_vga
#endif
#endif

#ifdef MXMODULE_DRIVER_13H
#ifndef MX__DRIVER_DEFAULT
#define MX__DRIVER_DEFAULT mx_driver_13h
#endif
#endif

#ifdef __cplusplus
}
#endif
#endif
